/************************************************************************************
* This module is the PHY layer Radio(interrupt) handlers
*
* (c) Copyright 2009, Freescale, Inc.  All rights reserved.
*
* No part of this document must be reproduced in any form - including copied,
* transcribed, printed or by any electronic means - without specific written
* permission from Freescale Semiconductor.
*
************************************************************************************/

#include "EmbeddedTypes.h"
#include "TransceiverDrv.h"
#include "PortConfig.h"
#include "Interrupt.h"
#include "Phy.h"

#include "Led.h"

#include "crc_ibm.h" 

#include "DataWhitening.h" 

#include "UartUtil.h"

#include "Utilities_Interface.h" 

/************************************************************************************
*************************************************************************************
* Private functions
*************************************************************************************
************************************************************************************/

//extern void PhyPlmeB2BIndication(void);

/************************************************************************************
*************************************************************************************
* Private memory declarations
*************************************************************************************
************************************************************************************/

void (* gpfPendingRxEvent)(void);
void (* gpfPendingTxEvent)(void);

static uint8_t mPhyState = gIdle_c;
static phyTxIsrParameters_t mPhyTxIsrParams;
static phyRxIsrParameters_t mPhyRxIsrParams;
static phyCcaIsrParameters_t mPhyCcaIsrParams;

uint16_t   u16ReceivedPackets;   //For Rx Per
uint16_t   u16PacketsIndex;      //For Rx Per 


/************************************************************************************
*************************************************************************************
* Private functions
*************************************************************************************
************************************************************************************/

/******************************************************************************
* Description :
*               
* Assumptions : 
*               
* Inputs      : 
* Output      : 
* Errors      : 
******************************************************************************/

void PhyPassCcaParams(uint8_t ccaParam)
{
  mPhyCcaIsrParams.flags = 0;
  mPhyCcaIsrParams.rssiAcc = 0;
  mPhyCcaIsrParams.ccaRssi = 0;
  mPhyCcaIsrParams.ccaParam = ccaParam;
  mPhyCcaIsrParams.ccaMeasurementsCnt = gPhyPib.pPIBphyRfConstants->numRssiMeasurements;
  mPhyCcaIsrParams.ccaThreshold = gPhyPib.pPIBphyRfConstants->ccaThreshold;
}

/******************************************************************************
* Description :
*               
* Assumptions : 
*               
* Inputs      : 
* Output      : 
* Errors      : 
******************************************************************************/

void PhyPassRxParams(phyPacket_t *pRxData, phyRxParams_t *pRxParams, uint8_t enableAckFilter)
{
  //pass RX PHY parameters to the transceiver ISR
  mPhyRxIsrParams.flags = 0;
  mPhyRxIsrParams.frameLength = 0;
  mPhyRxIsrParams.remainingLength = 0;
  mPhyRxIsrParams.macDataIndex = 0;
  mPhyRxIsrParams.unfillFifoBlockLength = 0;
  mPhyRxIsrParams.pRxPacket = pRxData;
  mPhyRxIsrParams.pRxParams = pRxParams;
  if(enableAckFilter)
  {
    mPhyRxIsrParams.flags |= PhyRxFlags_EnableAckFilter;
  }
}

/******************************************************************************
* Description :
*               
* Assumptions : 
*               
* Inputs      : 
* Output      : 
* Errors      : 
******************************************************************************/

uint8_t PhyPassTxParams(phyPacket_t *pTxPacket)
{
  uint8_t fifoThreshold = gFifoThreshold_c;
  /* pass TX PHY parameters to the transceiver ISR */
  mPhyTxIsrParams.pTxPacket = pTxPacket;
  mPhyTxIsrParams.flags = 0;
  mPhyTxIsrParams.macDataIndex = 0;
  mPhyTxIsrParams.remainingLength = pTxPacket->frameLength; 
  mPhyTxIsrParams.fillFifoBlockLength = gFillFifoBlockLength_c;
  
  /* if frame length is less than FillFifoBlock */
  if(mPhyTxIsrParams.remainingLength < gFillFifoBlockLength_c)
  {
    /*... adjust FillFifoBlock and FifoThreshold */
    mPhyTxIsrParams.fillFifoBlockLength = mPhyTxIsrParams.remainingLength;
    fifoThreshold = mPhyTxIsrParams.remainingLength - 1;
  }
  
  return fifoThreshold;
}

/******************************************************************************
* Description :
*               
* Assumptions : 
*               
* Inputs      : 
* Output      : 
* Errors      : 
******************************************************************************/

uint8_t PhyGetState(void)
{
        uint8_t phyState;	
        uint32_t CCR;
	
#if defined(IAR)	
  CCR=  IntDisableAll();	
#elif defined (CW)	
	IntDisableAll(CCR);
#endif
	
  phyState = mPhyState;
  IntRestoreAll(CCR);
  return phyState;
}

/******************************************************************************
* Description :
*               
* Assumptions : 
*               
* Inputs      : 
* Output      : 
* Errors      : 
******************************************************************************/

void PhySetState(uint8_t phyState)
{
  uint32_t CCR;
  
  
#if defined(IAR)	
  CCR=  IntDisableAll();	
#elif  defined (CW)	
	IntDisableAll(CCR);
#endif
	
  mPhyState = phyState;
  IntRestoreAll(CCR);
}

/******************************************************************************
* Description :
*               
* Assumptions : 
*               
* Inputs      : 
* Output      : 
* Errors      : 
******************************************************************************/

void PhyTxHandleDummyEvent(void)
{
  //PhyPlmeUnlockTx(); 						                //B36846 (Line originally commented out) Fcn is empty
}

/******************************************************************************
* Description :
*               
* Assumptions : 
*               
* Inputs      : 
* Output      : 
* Errors      : 
******************************************************************************/

void MKW01Drv_FillFifo(void)
{
  uint8_t blockLength = mPhyTxIsrParams.fillFifoBlockLength;
  uintn8_t blockIndex;
  uint8_t *blockPtr = (uint8_t *)&mPhyTxIsrParams.pTxPacket->data[mPhyTxIsrParams.macDataIndex];
  
  if(mPhyTxIsrParams.remainingLength <= gFillFifoBlockLength_c)
  {
    blockLength = mPhyTxIsrParams.remainingLength;
  }
  
  AssertSS;
  
  SpiRegs->DL = (unsigned char)(0x00 | 0x80);                                   
  for(blockIndex = 0; blockIndex < blockLength; blockIndex++)
  {
    WaitSPITxBufferEmpty;
    SpiRegs->DL = *blockPtr;                                                    
    blockPtr++;
  }
  
  //WaitSPITransmitterEmpty;       
  WaitSPITxBufferEmpty;        
  SPID_DummyRead;
  WaitSPIRxBufferFull;         
  
  DeAssertSS;
  
  mPhyTxIsrParams.remainingLength -= blockLength;
  mPhyTxIsrParams.macDataIndex += blockIndex;

}

/******************************************************************************
* Description :
*               
* Assumptions : 
*               
* Inputs      : 
* Output      : 
* Errors      : 
******************************************************************************/

void MKW01Drv_UnFillFifo(void)
{
  uint8_t blockLength = mPhyRxIsrParams.unfillFifoBlockLength;
  uintn8_t blockIndex;
  uint8_t *blockPtr = (uint8_t *)&mPhyRxIsrParams.pRxPacket->data[mPhyRxIsrParams.macDataIndex];
    
  AssertSS;                                                                     
  
  SpiRegs->DL = (unsigned char)(0x00);                                          
  WaitSPIRxBufferFull;
  for(blockIndex = 0; blockIndex < blockLength; blockIndex++)
  {
    SpiRegs->DL = 0;                                                            
    WaitSPIRxBufferFull;
    *blockPtr = SpiRegs->DL;                                                    
    blockPtr++;
  }
  
  DeAssertSS
  
  mPhyRxIsrParams.remainingLength -= blockLength;
  mPhyRxIsrParams.macDataIndex += blockIndex;

}

/******************************************************************************
* Description :
*               
* Assumptions : 
*               
* Inputs      : 
* Output      : 
* Errors      : 
******************************************************************************/

void PhyTxPacketSentEvent(void)
{
  
  mPhyTxIsrParams.flags |= PhyTxFlags_PacketSent;
  //PacketSentMask
  MKW01Drv_IRQ_DIO0_Disable();  
  gpfPendingTxEvent = PhyTxHandleDummyEvent; 
  PhySetState(gIdle_c);
  MKW01Drv_WriteRegister(MKW01_Reg_OpMode, (uint8_t) ( OpMode_Sequencer_On | OpMode_Listen_Off | OpMode_StandBy )); //StandBy
  
  PhyPdDataConfirm();

}

/******************************************************************************
* Description :
*               
* Assumptions : 
*               
* Inputs      : 
* Output      : 
* Errors      : 
******************************************************************************/

void PhyTxHandleFifoLevelEvent(void)
{
  if(mPhyTxIsrParams.remainingLength > 0)
  {
    MKW01Drv_FillFifo();
    gpfPendingTxEvent = PhyTxHandleFifoLevelEvent;
  }
  else
  {
    gpfPendingTxEvent = PhyTxPacketSentEvent; 
    //FifoLevelMask
    MKW01Drv_IRQ_DIO1_Disable();              
  
  }
}

/******************************************************************************
* Description :
*               
* Assumptions : 
*               
* Inputs      : 
* Output      : 
* Errors      : 
******************************************************************************/


void PhyRxHandleDummyEvent(void)
{
  ;
}

/******************************************************************************
* Description :
*               
* Assumptions : 
*               
* Inputs      : 
* Output      : 
* Errors      : 
******************************************************************************/

void PhyRxHandlePayloadReadyEvent(void)
{
  uint16_t CRC_verification_Rx; //Variable where CRC verefication is going to be stored
  uint8_t CRC_verificationMSB_Rx, CRC_verificationLSB_Rx;  // LSB and MSB of CRC verification
  uint8_t lastPayloadByte/*, irqFlags2  Original Line Commented. It was used for CRC by HW*/;
    
  /*MKW01Drv_ReadRegisterFastMacro(MKW01_Reg_IrqFlags2, irqFlags2); Original line commented. It was used for CRC by HW*/
  MKW01Drv_ReadRegisterFastMacro(MKW01_Reg_Fifo, lastPayloadByte);
  
  mPhyRxIsrParams.pRxPacket->data[mPhyRxIsrParams.macDataIndex] = lastPayloadByte;
  mPhyRxIsrParams.macDataIndex++;
  mPhyRxIsrParams.remainingLength--;
  
  
  //PayloadReadyMask
  MKW01Drv_IRQ_DIO0_Disable(); 
  
  gpfPendingRxEvent = PhyRxHandleDummyEvent;

  PhySetState(gIdle_c);
  MKW01Drv_WriteRegisterFastMacro(MKW01_Reg_OpMode, (uint8_t) ( OpMode_Sequencer_On | OpMode_Listen_Off | OpMode_StandBy )); //StandBy
/*  
  if(!(irqFlags2 & 0x02))                                                       // Original lines commented. This was used to CRC by HW
  {
    mPhyRxIsrParams.flags |= PhyRxFlags_CrcError;
    PhyPlmeRxCRCErrorIndication();
  }
*/  
  
 /* for (PhyIndex = 0; PhyIndex < 19; PhyIndex++){
      PhyMyRxPacket [PhyIndex] = mPhyRxIsrParams.pRxPacket->data [PhyIndex+1];  
  }*/
  

  //RadioComputeWhitening(&mPhyRxIsrParams.pRxPacket->frameLength, 20 );          // De-whitening IBM data received
  RadioComputeWhitening_CCIT(&mPhyRxIsrParams.pRxPacket->frameLength, 20 );          // De-whitening CCIT data received
  
  /*CRC calculation by SW*/
  CRC_verification_Rx = RadioComputeCRC(&mPhyRxIsrParams.pRxPacket->frameLength, 18, CRC_TYPE_IBM); // Calculate CRC from the first 18 payload bytes (2 last bytes contains CRC calculation from TX)
  CRC_verificationMSB_Rx = (uint8_t) (CRC_verification_Rx >> 8);                // CRC verification MSB 
  CRC_verificationLSB_Rx = (uint8_t) CRC_verification_Rx;                       // CRC verification LSB 
  if((!(CRC_verificationMSB_Rx == mPhyRxIsrParams.pRxPacket->data[17])) || (!(CRC_verificationLSB_Rx == mPhyRxIsrParams.pRxPacket->data[18]))) // If invalid CRC
  {
    mPhyRxIsrParams.flags |= PhyRxFlags_CrcError;
    
    u16PacketsIndex++;                                                          // Increment index for a packet received. PER RX
    
    LED_ToggleLed(gLed3_c);                                                     // Toggle led 3 to indicate a packet was received with an invalid CRC
        
    PhyPlmeRxCRCErrorIndication();                                              // Indicate an invalid CRC

  }
  else                                                                          // If valid CRC
  {
    mPhyRxIsrParams.flags |= PhyRxFlags_RxDone;

    u16PacketsIndex++;                                                          // Increment index for a packet received. PER RX
        
    u16ReceivedPackets++;                                                       // Increment packets received successfully variable. PER RX
   
    LED_ToggleLed(gLed1_c);                                                     // Indicate a valid CRC
    
    PhyPdDataIndication();
  }
}

/******************************************************************************
* Description :
*               
* Assumptions : 
*               
* Inputs      : 
* Output      : 
* Errors      : 
******************************************************************************/

void PhyRxHandleFifoLevelEvent(void)
{
  uint8_t readData;
  uint8_t frameType;
  
  if(mPhyRxIsrParams.remainingLength > 0)
  {
    MKW01Drv_ReadRegisterFastMacro(MKW01_Reg_Fifo, readData);
    
    mPhyRxIsrParams.pRxPacket->data[mPhyRxIsrParams.macDataIndex] = readData;
    mPhyRxIsrParams.macDataIndex++;
    mPhyRxIsrParams.remainingLength--;
    
    gpfPendingRxEvent = PhyRxHandleFifoLevelEvent;
    //PhyRxHandleFifoLevelEvent(); 
    
    if((mPhyRxIsrParams.flags & PhyRxFlags_EnableAckFilter) && (mPhyRxIsrParams.macDataIndex == 1))
    {
       frameType = readData & (0x07);
       if((frameType != 0x02) || (mPhyRxIsrParams.pRxPacket->frameLength != 3))
       {
          mPhyRxIsrParams.flags |= PhyRxFlags_InvalidLength;
              /* Disable FifoLevelMask and SyncAddressMask interrupt */ 
          MKW01Drv_IRQ_Disable(); /*DIO0 and DIO1 disabled*/
      
          
          gpfPendingRxEvent = PhyRxHandleDummyEvent;
         
          PhySetState(gIdle_c);
          MKW01Drv_WriteRegisterFast(MKW01_Reg_OpMode, (uint8_t) ( OpMode_Sequencer_On | OpMode_Listen_Off | OpMode_StandBy )); //StandBy
          
          PhyPlmeRxPHRErrorIndication();
       }
    }

  }
  else
  {
   // gpfPendingRxEvent = PhyRxHandlePayloadReadyEvent; // Original line commented

    //FifoLevelMask
    MKW01Drv_IRQ_DIO1_Disable(); /*DIO1 disabled*/
    /*DIO0 Enabled, Rising Edge sentitivity*/
    /*PayloadReady (DIO0), Rising Edge*/
    MKW01_IRQ_DIO0_PCR |= PORT_PCR_ISF_MASK;                                      
    MKW01_IRQ_DIO0_PCR |= PORT_PCR_IRQC(9);                                     

    PhyRxHandlePayloadReadyEvent(); 
    
  }
}

/******************************************************************************
* Description :
*               
* Assumptions : 
*               
* Inputs      : 
* Output      : 
* Errors      : 
******************************************************************************/

void PhyRxHandlePHREvent(void)
{
  MKW01Drv_ReadRegisterFastMacro(MKW01_Reg_Fifo, mPhyRxIsrParams.remainingLength); 
  mPhyRxIsrParams.pRxPacket->frameLength = mPhyRxIsrParams.remainingLength; 
  mPhyRxIsrParams.remainingLength = mPhyRxIsrParams.remainingLength - 1;
  
  if((mPhyRxIsrParams.pRxPacket->frameLength > gPhyMaxDataLength_c) || (mPhyRxIsrParams.pRxPacket->frameLength < (gPhyMinDataLength_c - 2)))
  {
    mPhyRxIsrParams.flags |= PhyRxFlags_InvalidLength;
    
    /* Disable FifoLevelMask and SyncAddressMask interrupt */ 
    MKW01Drv_IRQ_Disable();                                                     //DIO0 and DIO1 IRQ Disabled
    gpfPendingRxEvent = PhyRxHandleDummyEvent;
    PhySetState(gIdle_c);
    MKW01Drv_WriteRegisterFast(MKW01_Reg_OpMode, (uint8_t) ( OpMode_Sequencer_On | OpMode_Listen_Off | OpMode_StandBy )); //StandBy    
    PhyPlmeRxPHRErrorIndication();
  }
  else
  {
    mPhyRxIsrParams.flags |= PhyRxFlags_PhrReceived;
    
    gpfPendingRxEvent = PhyRxHandleFifoLevelEvent;
    
  }
}

/******************************************************************************
* Description :
*               
* Assumptions : 
*               
* Inputs      : 
* Output      : 
* Errors      : 
******************************************************************************/

void PhyRxHandleSyncAddresEvent(void)
{
  uint16_t lqiAcc = 0;
  
  mPhyRxIsrParams.flags |= PhyRxFlags_SfdReceived;

#if 1  
  MKW01Drv_WriteRegister(MKW01_Reg_RssiConfig, 0x01);

  if( 0x30 != MKW01Drv_ReadRegister(MKW01_Reg_TestDagc))                        
  {
   while((MKW01Drv_ReadRegister(MKW01_Reg_RssiConfig) & 0x02) == 0);            //DAGC, Low Beta Index. RssiDone Flag only sets when MKW01_Reg_TestDagc==0x10
  }
  lqiAcc += (uint16_t) MKW01Drv_ReadRegister(MKW01_Reg_RssiValue);
  
  MKW01Drv_WriteRegister(MKW01_Reg_RssiConfig, 0x01);

  if( 0x30 != MKW01Drv_ReadRegister(MKW01_Reg_TestDagc))                        
  {
   while((MKW01Drv_ReadRegister(MKW01_Reg_RssiConfig) & 0x02) == 0);            //DAGC, Low Beta Index. RssiDone Flag only sets when MKW01_Reg_TestDagc==0x10
  }
  lqiAcc += (uint16_t) MKW01Drv_ReadRegister(MKW01_Reg_RssiValue);
  
  mPhyRxIsrParams.pRxParams->linkQuality = (uint8_t) (lqiAcc >> 2);
#endif  

    //PayloadReady -> DIO0 (KBI2 PIN 0), FifoLevel -> DIO1 (KBI2 PIN 1), Timeout -> DIO4 (KBI2 PIN 4)
  MKW01Drv_ConfigureDIOPinsFast((DIO0_RxPayloadReady | DIO1_RxFifoLevel | DIO2_RxLowBat | DIO3_RxFifoFull),(DIO4_RxTimeout | DIO5_RxClkOut));
  //SyncAddressMask
  MKW01Drv_IRQ_DIO0_Disable();
  
  gpfPendingRxEvent = PhyRxHandlePHREvent;
  
  PhyPlmeRxSFDDetectIndication();

}

/******************************************************************************
* Description :
*               
* Assumptions : 
*               
* Inputs      : 
* Output      : 
* Errors      : 
******************************************************************************/

void PhyTimeHandleEventTimeout(void)
{
  
  //Put transceiver in Stand By mode
  MKW01Drv_WriteRegister(MKW01_Reg_OpMode, (uint8_t) ( OpMode_Sequencer_On | OpMode_Listen_Off | OpMode_StandBy )); //StandBy
  
  switch(mPhyState)
  {
    case gIdle_c:
      PhySetState(gIdle_c);
      PhyTimeTimeoutIndication();       				        //B36846 (Line originally commented out)
    break;    
    case gTX_c:
    {
      mPhyTxIsrParams.flags |= PhyTxFlags_TxTimeout;
      gpfPendingTxEvent = PhyTxHandleDummyEvent;
      /* Disable MKW01FifoLevelMask and MKW01SyncAddressMask interrupt */ 
      MKW01Drv_IRQ_Disable();                                                   //DIO0, DIO1 IRQ disabled
      PhySetState(gIdle_c);
      PhyTimeTimeoutIndication(); 						//B36846 (Line originally commented out)
      break;
    }
    case gRX_c:
    {
      mPhyRxIsrParams.flags |= PhyRxFlags_RxTimeout;
      gpfPendingRxEvent = PhyRxHandleDummyEvent;
      /* Disable MKW01FifoLevelMask and MKW01SyncAddressMask interrupt */ 
      MKW01Drv_IRQ_Disable();                                                   //DIO0, DIO1 IRQ disabled
      PhySetState(gIdle_c);
      PhyTimeTimeoutIndication(); 						//B36846 (Line originally commented out)
      break;
    }
    case gCCA_c:
    {
      PhyPlmeCcaConfirm(gChannelIdle_c); 				        //B36846 (Line originally commented out)
      mPhyCcaIsrParams.flags |= PhyCcaFlags_CcaTimeout;
      //Drv_IRQPinDisable();
      break;
    }
    case gED_c:
    {
      PhySetState(gIdle_c);
      PhyPlmeEdConfirm((uint8_t) 0xFF);
      mPhyCcaIsrParams.flags |= PhyCcaFlags_CcaTimeout;
      //Drv_IRQPinDisable();
      break;
    }
    default:
    {
      PhySetState(gIdle_c);
    }
  }
}

/******************************************************************************
* Description :
*               
* Assumptions : 
*               
* Inputs      : 
* Output      : 
* Errors      : 
******************************************************************************/

void PhyTimeHandleEventTrigger(void)
{
  switch(mPhyState)
  {
    case gIdle_c:
    // PhyPlmeB2BIndication(); 							//B36846 (Line originally commented out) Fcn is empty
    break;
    case gTX_c:
    {
      /* Put transceiver in TX mode */
      MKW01Drv_WriteRegister(MKW01_Reg_OpMode, (uint8_t) ( OpMode_Sequencer_On | OpMode_Listen_Off | OpMode_Transmitter)); //TX
      break;
    }
    case gRX_c:
    case gCCA_c:
    case gED_c:
    {
     //Put transceiver in RX mode
      MKW01Drv_WriteRegister(MKW01_Reg_OpMode, (uint8_t) ( OpMode_Sequencer_On | OpMode_Listen_Off | OpMode_Receiver)); //RX 
      break;
    }
    default:
    {
    }
  }  
}

/******************************************************************************
* Description :
*               
* Assumptions : 
*               
* Inputs      : 
* Output      : 
* Errors      : 
******************************************************************************/

void MKW01HandlerInterrupt(void)
{
    MKW01Drv_IRQ_Clear();                                                       //Clear DIO0 or DIO1 (PortC pin interrupts)
    //Protect from Radio interrupts
    /* disable transceiver interrupts */
   //Disable DIO0,DIO1 and DIO4 port interrupts
  NVIC_ICER = 0x80000000;
   
    switch(mPhyState)
    {
      case gTX_c:
      {
        gpfPendingTxEvent();
        break;
      }
      case gRX_c:
      {
        gpfPendingRxEvent();
        break;
      }
      default:
      {
      }
    }
    
    //Unprotect from Radio interrupts
   //Re-enable PORTC and PORTD interrupts                                       //Same interrupt vector for PORTC and PORTD
    NVIC_ICPR |= 0x80000000;
    NVIC_ISER |= 0x80000000;
  }


/******************************************************************************
* Description :
*               
* Assumptions : 
*               
* Inputs      : 
* Output      : 
* Errors      : 
******************************************************************************/

void Phy_ISR(void)
{
  if((MKW01_IRQ_DIO0_PCR &PORT_PCR_ISF_MASK)||(MKW01_IRQ_DIO1_PCR &PORT_PCR_ISF_MASK))  
  {
    MKW01HandlerInterrupt();
  }
  else if (MKW01_IRQ_DIO4_PCR &PORT_PCR_ISF_MASK)
  {

    CCAPinHandler_Interrupt();
  }
  else
  {
  }
  
}

/******************************************************************************
* Description :
*               
* Assumptions : 
*               
* Inputs      : 
* Output      : 
* Errors      : 
******************************************************************************/

void CCAPinHandler_Interrupt(void)
{ 
 MKW01Drv_CCA_Clear();
  //Protect from Radio interrupts
  /* disable transceiver interrupts */
   //Disable DIO0,DIO1 and DIO4 port interrupts
 NVIC_ICER = 0x80000000;
   
  if((mPhyState == gCCA_c) || (mPhyState == gED_c))
  {
    if((mPhyCcaIsrParams.flags & PhyCcaFlags_RxReadyReceived) != PhyCcaFlags_RxReadyReceived)//Rssi
    {
      mPhyCcaIsrParams.flags |= PhyCcaFlags_RxReadyReceived;
      EnableInterrupts();      // Enable other MCU irqs (e.g. application irqs). 
      while(mPhyCcaIsrParams.ccaMeasurementsCnt != 0)
      {
        MKW01Drv_WriteRegister(MKW01_Reg_RssiConfig, 0x01);

        if( 0x30 != MKW01Drv_ReadRegister(MKW01_Reg_TestDagc))                  
        {
        while((MKW01Drv_ReadRegister(MKW01_Reg_RssiConfig) & 0x02) == 0);       //DAGC, Low Beta Index. RssiDone Flag only sets when MKW01_Reg_TestDagc==0x10
        }  
        
        mPhyCcaIsrParams.rssiAcc += (uint16_t) MKW01Drv_ReadRegister(MKW01_Reg_RssiValue);
        mPhyCcaIsrParams.ccaMeasurementsCnt--;
      }
      mPhyCcaIsrParams.ccaRssi = (uint8_t) (mPhyCcaIsrParams.rssiAcc >> 3);
      mPhyCcaIsrParams.flags |= PhyCcaFlags_CcaDone;
      
      PhySetState(gIdle_c);
      //Put transceiver in Stand By mode
      MKW01Drv_WriteRegister(MKW01_Reg_OpMode, (uint8_t) ( OpMode_Sequencer_On | OpMode_Listen_Off | OpMode_StandBy )); //StandBy
      DisableInterrupts();    // Disable all interrupts from mcu
      
      if(mPhyCcaIsrParams.ccaParam == gCcaCCA_MODE1_c)
      {
        if(mPhyCcaIsrParams.ccaRssi > mPhyCcaIsrParams.ccaThreshold)
        {
           PhyPlmeCcaConfirm(gChannelIdle_c); 				        //B36846 (Line originally commented out)
        }
        else
        {
           PhyPlmeCcaConfirm(gChannelBusy_c); 				        //B36846 (Line originally commented out)
        }
      }
      else
      {
        PhyPlmeEdConfirm((uint8_t) mPhyCcaIsrParams.ccaRssi);
      }
      
    }
  }
  //Unprotect from Radio interrupts
   //Re-enable PORT C and PORT D interrupts
  NVIC_ICPR |= 0x80000000;
  NVIC_ISER |= 0x80000000;

}

